<?php
if (!defined('ABSPATH')) exit;

class Audit_Logger {

    // Create DB tables
    public static function create_tables() {
        global $wpdb;
        $charset_collate = $wpdb->get_charset_collate();

        $login_table = $wpdb->prefix . 'audit_trail_login_history';
        $activity_table = $wpdb->prefix . 'audit_trail_user_activities';

        require_once ABSPATH . 'wp-admin/includes/upgrade.php';

        dbDelta("CREATE TABLE $login_table (
            id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
            user_id BIGINT UNSIGNED NOT NULL,
            username VARCHAR(60),
            ip_address VARCHAR(100),
            browser TEXT,
            login_time DATETIME NULL,
            logout_time DATETIME NULL,
            status VARCHAR(20),
            message TEXT,
            PRIMARY KEY (id)
        ) $charset_collate;");

        dbDelta("CREATE TABLE $activity_table (
            id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
            user_id BIGINT UNSIGNED NOT NULL,
            role VARCHAR(50),
            datetime DATETIME NOT NULL,
            activity VARCHAR(100),
            type VARCHAR(50),
            description TEXT,
            PRIMARY KEY (id)
        ) $charset_collate;");
    }

    // Clean up old logs
    public static function delete_old_logs($days = 30) {
        global $wpdb;
        $login_table = $wpdb->prefix . 'audit_trail_login_history';
        $activity_table = $wpdb->prefix . 'audit_trail_user_activities';

        $cutoff = current_time('mysql', 1);
        $cutoff = date('Y-m-d H:i:s', strtotime("$cutoff -$days days"));

        $wpdb->query($wpdb->prepare("DELETE FROM $login_table WHERE login_time < %s", $cutoff));
        $wpdb->query($wpdb->prepare("DELETE FROM $activity_table WHERE datetime < %s", $cutoff));
    }

    // Hook events
    public static function hook_events() {
        add_action('wp_login', [__CLASS__, 'log_login'], 10, 2);
        add_action('clear_auth_cookie', [__CLASS__, 'log_logout']);
        add_action('wp_login_failed', [__CLASS__, 'log_failed_login']);
        add_action('transition_post_status', [__CLASS__, 'log_post_changes'], 10, 3);
        add_action('profile_update', [__CLASS__, 'log_profile_update'], 10, 2);
        add_action('user_register', [__CLASS__, 'log_user_register']);
        add_action('deleted_user', [__CLASS__, 'log_user_delete']);
        add_action('activated_plugin', [__CLASS__, 'log_plugin_change']);
        add_action('deactivated_plugin', [__CLASS__, 'log_plugin_change']);
        add_action('switch_theme', [__CLASS__, 'log_theme_change']);
    }

    public static function log_login($user_login, $user) {
        global $wpdb;
        $table = $wpdb->prefix . 'audit_trail_login_history';
        $ip = $_SERVER['REMOTE_ADDR'] ?? '';
        $browser = $_SERVER['HTTP_USER_AGENT'] ?? '';
        $now = current_time('mysql');

        $wpdb->insert($table, [
            'user_id'    => $user->ID,
            'username'   => $user->user_login,
            'ip_address' => $ip,
            'browser'    => $browser,
            'login_time' => $now,
            'status'     => 'Login',
            'message'    => 'Logged in successfully',
        ]);
    }

    public static function log_logout() {
        if (!is_user_logged_in()) return;
        $user = wp_get_current_user();
        if (!$user || $user->ID === 0) return;

        global $wpdb;
        $table = $wpdb->prefix . 'audit_trail_login_history';

        $row = $wpdb->get_row($wpdb->prepare("SELECT id FROM $table WHERE user_id = %d AND logout_time IS NULL ORDER BY login_time DESC LIMIT 1", $user->ID));

        if ($row) {
            $wpdb->update(
                $table,
                ['logout_time' => current_time('mysql')],
                ['id' => $row->id],
                ['%s'],
                ['%d']
            );
        }
    }

    public static function log_failed_login($username) {
        global $wpdb;
        $table = $wpdb->prefix . 'audit_trail_login_history';
        $ip = $_SERVER['REMOTE_ADDR'] ?? '';
        $browser = $_SERVER['HTTP_USER_AGENT'] ?? '';
        $now = current_time('mysql');

        $message = username_exists($username) ? "Failed login attempt" : "Failed login attempt with username Unknown username";

        $wpdb->insert($table, [
            'user_id'    => 0,
            'username'   => sanitize_text_field($username),
            'ip_address' => $ip,
            'browser'    => $browser,
            'login_time' => $now,
            'status'     => 'Failed Login',
            'message'    => $message,
        ]);
    }

    public static function log_post_changes($new_status, $old_status, $post) {
        if ($new_status === $old_status) return;

        $user = wp_get_current_user();
        if (!$user->ID) return;

        $action = match ($new_status) {
            'publish' => 'Published',
            'draft' => 'Moved to Draft',
            'trash' => 'Trashed',
            default => 'Updated',
        };

        self::log_activity([
            'user_id' => $user->ID,
            'role' => implode(', ', $user->roles),
            'activity' => "$action Post",
            'type' => $post->post_type,
            'description' => "Post titled '{$post->post_title}' was {$action}.",
        ]);
    }

    public static function log_profile_update($user_id, $old_user_data) {
        $user = get_userdata($user_id);
        if (!$user) return;

        $changes = [];

        if ($old_user_data->user_email !== $user->user_email) {
            $changes[] = "Email changed to {$user->user_email}";
        }
        if ($old_user_data->display_name !== $user->display_name) {
            $changes[] = "Display name changed to {$user->display_name}";
        }
        if (!empty($_POST['pass1'])) {
            $changes[] = "Password changed";
        }
        if ($old_user_data->roles !== $user->roles) {
            $changes[] = "Role changed to " . implode(', ', $user->roles);
        }

        if (!empty($changes)) {
            self::log_activity([
                'user_id' => $user_id,
                'role' => implode(', ', $user->roles),
                'activity' => 'Updated Profile',
                'type' => 'user',
                'description' => implode(', ', $changes),
            ]);
        }
    }

    public static function log_user_register($user_id) {
        $user = get_userdata($user_id);
        self::log_activity([
            'user_id' => $user_id,
            'role' => implode(', ', $user->roles),
            'activity' => 'Registered',
            'type' => 'user',
            'description' => "New user registered: {$user->user_login}",
        ]);
    }

    public static function log_user_delete($user_id) {
        $user = get_userdata($user_id);
        self::log_activity([
            'user_id' => $user_id,
            'role' => implode(', ', $user->roles),
            'activity' => 'Deleted User',
            'type' => 'user',
            'description' => "User account deleted: {$user->user_login}",
        ]);
    }

    public static function log_plugin_change($plugin) {
        $user = wp_get_current_user();
        if (!$user->ID) return;

        $status = current_filter() === 'activated_plugin' ? 'Activated Plugin' : 'Deactivated Plugin';

        self::log_activity([
            'user_id' => $user->ID,
            'role' => implode(', ', $user->roles),
            'activity' => $status,
            'type' => 'plugin',
            'description' => "Plugin {$plugin} was {$status}.",
        ]);
    }

    public static function log_theme_change($theme_name) {
        $user = wp_get_current_user();
        if (!$user->ID) return;

        self::log_activity([
            'user_id' => $user->ID,
            'role' => implode(', ', $user->roles),
            'activity' => 'Switched Theme',
            'type' => 'theme',
            'description' => "Theme changed to {$theme_name}.",
        ]);
    }

    private static function log_activity($data) {
        global $wpdb;
        $table = $wpdb->prefix . 'audit_trail_user_activities';

        $wpdb->insert($table, [
            'user_id' => $data['user_id'],
            'role' => $data['role'],
            'datetime' => current_time('mysql'),
            'activity' => $data['activity'],
            'type' => $data['type'],
            'description' => $data['description'],
        ]);
    }
}
